/*! \file Installdisk.h
*  \author Laszlo Federics
*  \date   28.04.2011
*  \brief  Class definition file for class Installdisk and InstalldiskArguments
*
*  This file is part of the Installdisk tool.
*
*  (C) Copyright Siemens AG A&D MC 2011. All rights reserved.
*/


#ifndef INSTALLDISK_H
#define INSTALLDISK_H

#include <windows.h>
#include <winioctl.h>


# if !defined(INSTALLDISK_EXPORT)
#   if defined(WIN32) || defined(WIN64)
#     ifdef INSTALLDISK_EXPORTS
#       define INSTALLDISK_EXPORT __declspec(dllexport)
#     else
#       define INSTALLDISK_EXPORT __declspec(dllimport)
#     endif
#   else
#     define INSTALLDISK_EXPORT
#   endif
# endif


/*! \class InstalldiskArguments
*  \brief A class for input arguments for the Installdisk class.
*
*  The class stores the input arguments of the Installdisk class. The parameters
*  are public, and can be set freely.\n
*  The method Initialize supports the processing of command line arguments of 
*  the following syntax:
*
*  - SYNOPSIS\n
*     - [options] image-file drive-letter\n
*  \n
*  - DESCRIPTION\n
*     - INSTALLBOOT\n
*        - Install or extract the boot loader of a disk drive.\n
*     - INSTALLDISK\n
*        - Create or install a physical image of a disk drive\n
*          or of a single partition.\n
*  \n
*  - OPERANDS\n
*     - image-file\n
*        - The filename of the image.\n
*  \n
*     - drive-letter\n
*        - The name of the disk drive onto which the image\n
*          is to be installed or from which it is extracted.\n
*  \n
*  - OPTIONS\n
*     - --help\n
*        - Print this help message.\n
*  \n
*     - --verbose\n
*        - Print verbose messages.\n
*  \n
*     - --dry-run\n
*        - Do not modify disk drive.\n
*  \n
*     - --force\n
*        - Normally, operation is restricted to access removable\n
*          media only, but if this option is specified, all hard\n
*          disk drives are allowed.\n
*  \n  
*        - WARNING: This option is for expert users only!\n
*          YOU MAY DAMAGE YOUR WINDOWS INSTALLATION!\n
*  \n
*    INSTALLBOOT\n
*     - --boot-cu320\n
*        - Use parameters for CU320 boot loader rather than for D435.\n
*          But see restrictions.\n
*  \n
*     - --boot-num-sectors <number>\n
*        - Optionally specify the length of the bootloader. The special\n
*          value of 0 means use as much space as available in front of\n
*          the partition with the lowest start sector.\n
*          But see restrictions.\n
*   \n
*     - --install-partition-table\n
*        - Writes the partition table to disk drive if it was part\n
*          of the image.\n
*  \n
*    INSTALLDISK\n
*     - --disk-partition <number>\n
*        - Install image to a specific disk partition. The partition\n
*          numbers are counted starting with 1. 1-4 are reserved for\n
*          the primary partitions. from 5 onwards are  the extended\n
*          partitions.\n
*  \n
*     - --blocksize <size>[b|k|m]\n
*        - Blocksize used for transfer. The numerical size value may\n
*          be extended by a multiply-letter b|k|m for sizes in bytes,\n
*          kilobytes or megabytes. The blocksize should be a muliply\n
*          of the physical sector size of the media and is used for\n
*          increasing the tranfer speed. The default size is 512 bytes.\n
*  \n
*     - --extract\n
*        - Do not install, but read image from disk drive to image\n
*          file. The image file must not already exist! If it already\n
*          exists, an error message is displayed and the program\n
*          terminates.\n
*  \n
*  - RESTRICTIONS\n
*    - You will need admin permissions in order to run this\n
*      admininistrator program.\n
*  \n
*    INSTALLBOOT\n
*    - The option --boot-cu320 is provided in the hope that it is helpful\n
*      but it must be used with care.\n
*  \n
*    - --boot-cu320 only for installing boot loaders\n
*      from released image files, you will be safe.\n
*  \n
*    - When used in combination with --extract, however, it may produce\n
*      an incomplete boot loader because the size of the boot loader on a\n
*      CU320 disk must be estimated:\n
*  \n
*     -   1) If the estimated size is too small, the extracted boot\n
*            loader is incomplete and if you use such an incomplete\n
*            image-file to install a boot loader, the loader will not\n
*            work.\n
*  \n
*     -   2) If the estimated size is too large, the extracted boot\n
*            loader will contain garbage at the end of image-file\n
*            and if you use such an overly long image-file to install\n
*            a boot loader, the start of the first partition on the\n
*            disk can be overwritten.\n
*  \n
*  - EXAMPLES\n
*  \n
*            - INSTALLBOOT\n
*     - To install the boot block in file \"boot.bin\" to the boot sector\n
*       of removable media drive F: use\n
*  \n
*        - % installboot boot.bin F:\n
*  \n
*     - To save the boot sector of drive F: into file \"boot.bin\" use\n
*  \n
*        - % installboot --extract boot.bin F:\n
*  \n
*     - To install a boot sector of drive F: into file \"boot.bin\" allowed\n
*       to take as much free space as can be found in front of the first\n
*       partition use\n
*  \n
*        - % installboot --boot-num-sectors 0 boot.bin F:\n
*  \n
*            - INSTALLDISK\n
*     - To install the disk image file \"disk.bin\" to the removable\n
*       media drive F: use\n
*  \n      
*        - % installdisk disk.bin F:\n
*  \n
*     - To install an image with increased block transfer buffers\n
*       of 512 kbytes use\n
*  \n
*        - % installdisk --blocksize 512k disk.bin F:\n
*  \n
*     - To install an image with increased block transfer buffers\n
*       to the second primary partition and with progress messages\n
*       displayed use\n
*  \n
*        - % installdisk --verbose --disk-partition 2 \\n
*                        --blocksize 1m disk.bin F:\n
*  \n                 
*     - To save the physical image of drive F: into file \"disk.bin\" use\n
*  \n    
*        - % installdisk --extract disk.bin F:\n
*  \n
*/
class INSTALLDISK_EXPORT InstalldiskArguments
{
public:

  /*! \enum ErrorCodeEnum
  *
  *  The ErrorCode type defines the potential error values.
  */
  enum ErrorCodeEnum
  {
    /*! No errors. */
    OK,                                      
    
    /*! The number of boot sector is not specified. */
    BOOT_NUM_SECTOR_NUMBER_REQUIRED,         
    
    /*! The size of boot sector is not specified. */
    BOOT_NUM_SECTOR_SIZE_REQUIRED,           
    
    /*! The given partition number is invalid. */
    INVALID_BOOT_PARTITION_NUMBER,           
    
    /*! The number of disk partition is not specified. */    
    DISK_PARTITION_NUMBER_REQUIRED,      
    
    /*! The size of disk partition is not specified. */
    DISK_PARTITION_SIZE_REQUIRED,            
    
    /*! The given partition number is invalid. */
    INVALID_DISK_PARTITION_NUMBER,           
    
    /*! The blocksize is not specified. */     
    BLOCKSIZE_REQUIRED,                      
    
    /*! The format of the given blocksize multiplicator is invalid. */
    BLOCKSIZE_MULTIPLIKATOR_FORMAT_INVALID,  
    
    /*! The given multiplicator is not interpreted. */
    UNKNOWN_MULTIPLICATOR,                   
    
    /*! The given option is not interpreted. */
    UNKNOWN_OPTION,                          
    
    /*! The number of given arguments is not correct. */
    WRONG_NUMBER_OF_ARGUMENTS,               
    
    /*! The given driver letter is incorrect. */
    UNKNOWN_DRIVE_LETTER                 
  };


  /*! Flag of 'help' option */
  bool m_bIsHelp;  

  /*! Flag of 'dry-run' option */
  bool m_bIsDryRun;  

  /*! Flag of 'force' option */
  bool m_bIsForced;  

  /*! Flag of 'extract' option */
  bool m_bIsExtract;  

  /*! Flag of 'install-partition-table' option */
  bool m_bIsInstallPTable;  

  /*! Flag of the 'D435' bootloader */
  bool m_bIsBootD435;  

  /*! Flag of the 'CU320' bootloader */
  bool m_bIsBootCu320;  

  /*! Flag of 'verbose' option */
  bool m_bIsVerbose;  

  /*! Flag of 'boot-num-sect' option */
  bool m_bIsBootNumSect;  

  /*! Stored the specified blocksize */
  unsigned long m_ulBlockSize;  

  /*! Stored the number of specific selected partition */
  unsigned long m_ulDiskPartition;  

  /*! Optionally specify the length of the  bootloader */
  unsigned long m_ulBootNumSectors;  

  /*! Image filename */
  TCHAR m_strImageFileName[MAX_PATH];  

  /*! Drive letter of disk */
  TCHAR m_strDriveLetter[MAX_PATH]; 


  /*! \fn InstalldiskArguments()
  *
  * Default constructor to initialize all member varibales.
  */
  InstalldiskArguments();


  /*! \fn virtual ~InstalldiskArguments()
  *
  * Destructor.
  */
  virtual ~InstalldiskArguments();


  /*! \fn ErrorCodeEnum Initialize(int argc, TCHAR *argv[])
  *  \param argc Number of given parameters
  *  \param argv The array of arguments
  *  \retval ErrorCodeEnum Error code
  *
  * The function initializes the member variables according to the input
  * arguments. The syntax os these arguments can be found in the description
  * of the class.
  */
  ErrorCodeEnum Initialize(int argc, TCHAR *argv[]);

}; // class InstalldiskArguments



/*! \class Installdisk
*  \brief A class for the "installdisk" functionality
*
*  The class provides the installdisk functionality, without user
*  interface. Usage:
*  
*/
class INSTALLDISK_EXPORT Installdisk
{

public:

  /*! \enum ErrorCodeEnum
  *
  *  The ErrorCode type defines the possible error values.
  */
  enum ErrorCodeEnum
  {
    SUCCESS = 0,
    COULDNT_INIT_SHARED_MEMORY,
    COULDNT_CLOSE_SHARED_MEMORY,
    INVALID_COPY_BLOCK_SIZE,
    DISK_PARTITION_IS_UNASSIGNED,
    NOT_ENOUGH_SPACE_ON_DRIVE,
    IMAGEFILE_TOO_LARGE,
    MEDIATYPE_IS_NOT_REMOVABLE,
    ERROR_OPEN_SCMANAGER,
    ERROR_OPEN_SERVICE,
    ERROR_CLOSE_SERVICEHANDLE,
    ERROR_UNLOCK_VOLUME,
    ERROR_CLOSE_HANDLE,
    ERROR_WRITING_FILE_SECTOR,
    ERROR_READING_FILE_SECTOR,
    BLOCKSIZE_TOO_LARGE
  };


  /*! \enum Error
  *
  *  The structure contains the last windows error code (returned by 
  *  GetLastError()), and an error code of installdisk.
  */
  struct Error
  {
    //! The last windows error code
    DWORD dwWinError;

    //! The error code of Installdisk
    ErrorCodeEnum nError;

    //! True, if there is no error
    inline bool succeeded() {return ((dwWinError == ERROR_SUCCESS) && (nError == SUCCESS));}
  };


  struct PartitionTableEntry
  {
    unsigned char bootIndicator;
    unsigned char startSectorCHS[3];
    unsigned char partitionType;
    unsigned char endSectorCHS[3];
    unsigned int  startSectorLBA;
    unsigned int  lenLBA;
  };


  /*! \fn Installdisk()
  *
  * Default constructor to initialize all member varibales.
  */
  Installdisk();


  /*! \fn virtual ~Installdisk()
  *
  * Destructor.
  */
  ~Installdisk();


  /*! \fn Error Initialize(const InstalldiskArguments& arguments)
  *
  * Initialize the installdisk functionality. It means:
  * Opening the hard disk device with read or write access\n
  * Opening the binary image file with read or write access\n
  * Get the properties of the cf card and the partition\n
  * Lock and unmount the partition
  */
  Error Initialize(const InstalldiskArguments& arguments);


  /*! \fn Error CopyFromFileToCard(ULONG nAmountOfBytes, bool bCopyMbr)
  *
  * Copies the specified amount of bytes from the binary image file
  * to the cf card.
  */
  Error CopyFromFileToCard(ULONG nAmountOfBytes, bool bCopyMbr);


  /*! \fn Error CopyFromFileToCard(ULONG nAmountOfBytes, bool bCopyMbr)
  *
  * Copies the specified amount of bytes from the cf card
  * to the binary image file.
  */
  Error CopyFromCardToFile(ULONG nAmountOfBytes, bool bCopyMbr);


  /*! \fn Error Finish()
  * Cleanup after the installdisk procedure.
  * Close the opened devices and files and release memory.
  */
  Error Finish();


  //! The number of sectors to copy
  ULONGLONG CopyNumSectors() const { return m_ullCopyNumSectors; }
  

  //! The number of bytes to copy in one loop
  ULONG CopyBlockSize() const { return m_ulCopyBlockSize; }
  

  //! Returns the number of bytes in one sector in the cf card
  DWORD BytesPerSector() const;


  //! Returns the first sector to write(read) of the cf card
  ULONGLONG DriveStartSector() const { return m_ullDriveStartSector; }
  
private:

  Installdisk::Error StartInstalldiskService();

  Installdisk::Error StopInstalldiskService();

  Installdisk::Error InitSharedMemory();

  Installdisk::Error CloseSharedMemory();

  DWORD ControlInstalldiskService(int nCommand);

  DWORD OpenImageFile(LPCTSTR strImageFile,
                      bool bIsExtract,
                      ULONGLONG* ullFileSize);

  DWORD ReadMbr(HANDLE hHandle);

  DWORD WriteMbr();

  DWORD ReadFileSector(HANDLE srcFileHandle,
                       char *sectorBufferP,
		                   ULONG nNrOfBytesToRead,
                       bool bExpectWholeSector);

  DWORD WriteFileSector(HANDLE destFileHandle,
                        char *sectorBufferP,
		                    DWORD nBytesToWrite);

  DWORD SeekFileSector(HANDLE fileHandle,
                       ULONGLONG sectorNumber,
		                   ULONG bytesPerSector);

  
  //! Handle to the service manager
  SC_HANDLE m_schSCManager;

  //! Handle to the installdisk service
  SC_HANDLE m_schService;

  //! Handle to file mapping object for shared memory
  HANDLE m_hMapFile;

  //! Pointer to the shared memory
  LPVOID m_pBuffer;

  //! input arguments
  InstalldiskArguments m_Arguments;

  //! access any kind of bootloader
  bool  m_bIsBoot;           
     
  //! start sector of hard disk device
  ULONGLONG m_ullDriveStartSector;   

  //! num sectors of hard disk device
  ULONGLONG m_ullDriveNumSectors;    

  //! num sectors to be transfered
  ULONGLONG m_ullCopyNumSectors;               

  //! blocksize used for copying
  ULONG m_ulCopyBlockSize;     

  //! size of the image in bytes
  ULONGLONG m_ullImageFileSize;   

  //! length of hard disk device
  ULONGLONG m_ullHardDiskLength;               

  //! expect a partition table on disk
  bool m_bIsPartitionTableExpected;

  /*! Handle to the image file. It is only used, if the InstalldiskSvc(which is 
      running) under SYSTEM, cannot open the file. Then the installdisk.dll must
      open the image file and use this handle.
   */
  HANDLE m_hImgDevice;

}; // class Installdisk


#endif //INSTALLDISK_H
